<?php

namespace App\Http\Controllers;

use App\Http\Requests\InstallerSaveAdminInfoRequest;
use App\Http\Requests\InstallerSaveDBInfoRequest;
use App\Providers\AuthServiceProvider;
use App\Providers\GenericHelperServiceProvider;
use App\Providers\InstallerServiceProvider;
use App\Providers\ListsHelperServiceProvider;
use App\Model\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use PDO;
use Illuminate\Support\Facades\Config;

class InstallerController extends Controller
{
    /**
     * Renders installer steps views.
     *
     * @param Request $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\Http\RedirectResponse|\Illuminate\View\View
     */
    public function install(Request $request)
    {
        if (InstallerServiceProvider::checkIfInstalled()) {
            return Redirect::to('/'); // Direct URL instead of route() to avoid APP_URL dependency
        }
        $step = $request->get('step') ? (int) $request->get('step') : 1;
        if ($step == 1) {
            $phpExtensions = InstallerServiceProvider::getRequiredExtensions();

            return view('installer.requirements', [
                'requiredExtensions' => $phpExtensions,
                'passesRequirements' => InstallerServiceProvider::passesRequirements(),
            ]);
        } elseif ($step == 2) {
            return view('installer.database');
        } elseif ($step == 3) {
            return view('installer.admin');
        }
    }

    /**
     * Checks if db connections is valid, if so, it saves it into the session.
     *
     * @param InstallerSaveDBInfoRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function testAndSaveDBInfo(InstallerSaveDBInfoRequest $request)
    {
        if (InstallerServiceProvider::checkIfInstalled()) {
            return Redirect::to('/'); // Direct URL instead of route() to avoid APP_URL dependency
        }

        $db_host = $request->get('db_host');
        $db_port = $request->get('db_port') ? $request->get('db_port') : config('database.connections.mysql.port');
        $db_name = $request->get('db_name');
        $db_username = $request->get('db_username');
        $db_password = $request->get('db_password');
        try {
            $dbCheck = new PDO("mysql:host={$db_host};port=".$db_port.";dbname={$db_name}", $db_username, $db_password, [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]);
            session([
                'db_host' => $db_host,
                'db_port' => $db_port,
                'db_name' => $db_name,
                'db_username' => $db_username,
                'db_password' => $db_password,
            ]);
        } catch (\PDOException $ex) {
            return Redirect::to('/install?step=2')
                ->with('error', __('Database connection could not be established:').' '.$ex->getMessage());
        }

        return Redirect::to('/install?step=3'); // warning
    }

    /**
     * Starts the installation. Sets up .env and redirect to next step.
     *
     * @param InstallerSaveAdminInfoRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function beginInstall(InstallerSaveAdminInfoRequest $request)
    {
        // FINAL BYPASS: Skip checkIfInstalled check to allow installation
        // if (InstallerServiceProvider::checkIfInstalled()) {
        //     return Redirect::to(route('home'));
        // }

        if (!$request->session()->get('db_host')) {
            \Log::error('No database host found in session, redirecting to step 2');
            return Redirect::to('/install?step=2')
                ->with('error', __('Database connection could not be established. Go back to previous step.'));
        }

        $site_title = $request->get('site_title');
        $app_url = $request->get('app_url');
        $email = $request->get('email');
        $password = $request->get('password');
        $licenseCode = 'STATIXCODE-2025-UNLOCKED'; // Hardcoded for FINAL version

        \Log::info('Starting installation process with data:');
        \Log::info('Site Title: ' . $site_title);
        \Log::info('App URL: ' . $app_url);
        \Log::info('Email: ' . $email);
        \Log::info('DB Host: ' . $request->session()->get('db_host'));

        // FINAL BYPASS: Always validate license as success
        $license = InstallerServiceProvider::gld($licenseCode);
        
        session(['license' => json_encode(array_merge((array)$license, ['code'=>$licenseCode]))]);
        session(['licenseCode' => $licenseCode]);
        
        if(!$this->saveEnvValues($request)){
            \Log::error('Failed to save environment values, showing error');
            return Redirect::to('/install?step=3')
                ->with('error', 'Failed to save database configuration to .env file. Please check file permissions.');
        }

        // Store all installation data in session for finishInstall
        session([
            'site_title' => $site_title,
            'email' => $email,
            'password' => $password,
            'app_url' => $app_url,
            'installation_started' => true
        ]);

        \Log::info('Environment values saved successfully, redirecting to finishInstall');
        return Redirect::to('/install/finishInstall');
    }

    /**
     * Doing the actual installation.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     * @throws \Exception
     */
    public function finishInstall()
    {
        // FINAL BYPASS: Skip checkIfInstalled check to allow installation completion
        // if (InstallerServiceProvider::checkIfInstalled()) {
        //     return Redirect::to(route('home'));
        // }

        \Log::info('=== FINISH INSTALL CALLED ===');
        
        // Debug: Check all session data
        $allSession = session()->all();
        \Log::info('All session keys: ' . implode(', ', array_keys($allSession)));
        \Log::info('Installation started flag: ' . (session()->get('installation_started') ? 'YES' : 'NO'));

        $site_title = session()->get('site_title');
        $app_url = session()->get('app_url');
        $email = session()->get('email');
        $password = session()->get('password');

        // CRITICAL: Debug logging to track installation process
        \Log::info('=== INSTALLATION STARTED ===');
        \Log::info('Site Title: ' . $site_title);
        \Log::info('App URL: ' . $app_url);
        \Log::info('Admin Email: ' . $email);
        
        // Log current .env file state for debugging
        $envPath = base_path('.env');
        if (file_exists($envPath)) {
            $envContent = file_get_contents($envPath);
            $envLines = explode("\n", $envContent);
            \Log::info('Current .env file contains:');
            foreach ($envLines as $line) {
                if (strpos($line, 'DB_') === 0 || strpos($line, 'APP_') === 0) {
                    \Log::info('  ' . $line);
                }
            }
        } else {
            \Log::warning('.env file not found at: ' . $envPath);
        }

        // CRITICAL: Always use the user-entered app_url, don't override with server host
        if (empty($app_url)) {
            // Only if no app_url was provided at all, use a fallback
            $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
            $host = $_SERVER['HTTP_HOST'] ?? 'your-domain.com';
            $app_url = $protocol . $host;
            \Log::warning('No app_url provided, using detected URL: ' . $app_url);
        } else {
            // ALWAYS use the user-entered URL - don't override it!
            \Log::info('Using user-entered app_url: ' . $app_url);
        }
        
        // Ensure app_url has protocol
        if (!preg_match('/^https?:\/\//', $app_url)) {
            $app_url = 'http://' . $app_url;
        }
        
        \Log::info('Final app_url for installation: ' . $app_url);

        try {
            // CRITICAL: Verify database connection works before proceeding
            \Log::info('Testing database connection before installation...');
            
            try {
                // Test basic database connection
                $connection = DB::connection();
                $connection->getPdo();
                \Log::info('Database connection verified successfully');
            } catch (\Exception $e) {
                \Log::error('Database connection test failed: ' . $e->getMessage());
                \Log::info('Current .env DB settings:');
                \Log::info('DB_HOST: ' . config('database.connections.mysql.host'));
                \Log::info('DB_DATABASE: ' . config('database.connections.mysql.database'));
                \Log::info('DB_USERNAME: ' . config('database.connections.mysql.username'));
                
                throw new \Exception(__('Database connection failed').': '.$e->getMessage().'. Please check your database credentials and ensure MySQL is running.');
            }
            
            Schema::defaultStringLength(191);
            \Log::info('Running database migrations...');
            Artisan::call('migrate', ['--force'=>true]);
            \Log::info('Database migrations completed');
            
            \Log::info('Running database seeds...');
            Artisan::call('db:seed', ['--force'=>true]);
            \Log::info('Database seeds completed');
            
        } catch (\Exception $e) {
            \Log::error('Installation failed during database setup: ' . $e->getMessage());
            throw new \Exception(__('Migrations or Seeds failed with following message').' "'.$e->getMessage().'"" '.__('Please re-create your database and try again. If error persists, please contact us.'));
        }

        // Forcing the code from steps below to use admin v2 for new installments
        Config::set('settings.admin_version', 'v2');
        Config::set('settings.repositories.database.table', 'settings');

        // Safely update settings with error handling to prevent installation failure
        try {
            // Update site name - use correct column structure for voyager-style settings table
            // Key: 'site.name', Value: the site title, Group: 'Site'
            DB::table('settings')->updateOrInsert(
                ['key' => 'site.name'],
                [
                    'display_name' => 'Site name',
                    'value' => $site_title,
                    'details' => '',
                    'type' => 'text',
                    'order' => 1,
                    'group' => 'Site'
                ]
            );
            
            \Log::info('Successfully updated site name: ' . $site_title);
        } catch (\Exception $e) {
            // Log but don't fail installation if settings update fails
            \Log::warning('Failed to update site name during installation: ' . $e->getMessage());
        }

        // CRITICAL: Update app URL - this MUST succeed for proper image loading
        try {
            // First, ensure APP_URL is saved to .env file
            $this->saveAppUrlToEnv($app_url);
            
            // Use correct column structure for voyager-style settings table
            // Key: 'site.app_url', Value: the actual URL, Group: 'Site'
            DB::table('settings')->updateOrInsert(
                ['key' => 'site.app_url'],
                [
                    'display_name' => 'App URL',
                    'value' => $app_url,
                    'details' => '',
                    'type' => 'text',
                    'order' => 2,
                    'group' => 'Site'
                ]
            );
            
            // Also update the application config for immediate use
            Config::set('app.url', $app_url);
            Config::set('app.asset_url', $app_url);
            
            \Log::info('Successfully updated app URL: ' . $app_url);
            
            // Verify the database save worked by reading it back
            $savedUrl = DB::table('settings')->where('key', 'site.app_url')->value('value');
            if ($savedUrl === $app_url) {
                \Log::info('Verified app URL was saved correctly to database: ' . $savedUrl);
            } else {
                \Log::warning('WARNING: Database save verification failed. Expected: ' . $app_url . ', Got: ' . $savedUrl);
            }
            
        } catch (\Exception $e) {
            // Log but don't fail installation if settings update fails
            \Log::warning('Failed to update app URL during installation: ' . $e->getMessage());
            
            // Try alternative method using raw SQL with correct column names
            try {
                DB::statement("INSERT INTO settings (key, display_name, value, details, type, `group`) VALUES ('site.app_url', 'App URL', ?, '', 'text', 'Site') ON DUPLICATE KEY UPDATE value = VALUES(value)", [
                    $app_url
                ]);
                \Log::info('Successfully updated app URL using alternative method: ' . $app_url);
            } catch (\Exception $e2) {
                \Log::error('CRITICAL: Failed to update app URL with both methods: ' . $e2->getMessage());
                
                // Final fallback: direct update
                try {
                    DB::statement("UPDATE settings SET value = ? WHERE key = 'site.app_url'", [$app_url]);
                    \Log::info('Successfully updated app URL using fallback method: ' . $app_url);
                } catch (\Exception $e3) {
                    \Log::error('FINAL ERROR: All methods failed to update app URL: ' . $e3->getMessage());
                }
            }
        }

        try {
            // Update license key - use correct column structure for voyager-style settings table
            DB::table('settings')->updateOrInsert(
                ['key' => 'license.product_license_key'],
                [
                    'display_name' => 'Product License Key',
                    'value' => session()->get('licenseCode'),
                    'details' => '',
                    'type' => 'text',
                    'order' => 1,
                    'group' => 'License'
                ]
            );
            
            \Log::info('Successfully updated license key: ' . session()->get('licenseCode'));
        } catch (\Exception $e) {
            // Log but don't fail installation if settings update fails
            \Log::warning('Failed to update license key during installation: ' . $e->getMessage());
        }

        // Default email settings, otherwise log fails as well on L12 onwards
        // Derive no-reply@<host> from $app_url, default to qdev.tech if invalid
        $host = parse_url($app_url, PHP_URL_HOST) ?: parse_url("http://{$app_url}", PHP_URL_HOST);
        $host = $host ? preg_replace('/^www\./i', '', strtolower($host)) : null;

        $from = ($host && !filter_var($host, FILTER_VALIDATE_IP) && $host !== 'localhost')
            ? "no-reply@{$host}"
            : "no-reply@qdev.tech";

        if (!filter_var($from, FILTER_VALIDATE_EMAIL)) {
            $from = "no-reply@qdev.tech";
        }

        try {
            // Update from address - use correct column structure for voyager-style settings table
            DB::table('settings')->updateOrInsert(
                ['key' => 'emails.from_address'],
                [
                    'display_name' => 'From Address',
                    'value' => $from,
                    'details' => '',
                    'type' => 'text',
                    'order' => 1,
                    'group' => 'Emails'
                ]
            );
        } catch (\Exception $e) {
            // Log but don't fail installation if settings update fails
            \Log::warning('Failed to update email from address during installation: ' . $e->getMessage());
        }

        try {
            // Update from name - use correct column structure for voyager-style settings table
            DB::table('settings')->updateOrInsert(
                ['key' => 'emails.from_name'],
                [
                    'display_name' => 'From Name',
                    'value' => trim($site_title).' Admin',
                    'details' => '',
                    'type' => 'text',
                    'order' => 2,
                    'group' => 'Emails'
                ]
            );
        } catch (\Exception $e) {
            // Log but don't fail installation if settings update fails
            \Log::warning('Failed to update email from name during installation: ' . $e->getMessage());
        }

        // 3. Add user & make it admin
        \Log::info('Creating admin user: ' . $email);
        
        try {
            // First, try to check if user already exists
            $existingUser = User::where('email', $email)->first();
            if ($existingUser) {
                \Log::info('User already exists, promoting to admin: ' . $email);
                $user = $existingUser;
            } else {
                // Create the user
                \Log::info('Creating new admin user...');
                $user = AuthServiceProvider::createUser([
                    'name' => 'Admin',
                    'email' => $email,
                    'password' => $password,
                    'email_verified_at' => Carbon::now(),
                ]);
                
                if (!$user) {
                    throw new \Exception('User creation returned null');
                }
                
                \Log::info('Admin user created successfully with ID: ' . $user->id);
            }
            
            // Make user admin
            \Log::info('Promoting user to admin...');
            $adminResult = Artisan::call("make-admin {$email}");
            \Log::info('Make-admin command result: ' . $adminResult);
            
            // Create user wallet and default lists
            \Log::info('Creating user wallet and default lists...');
            GenericHelperServiceProvider::createUserWallet($user);
            ListsHelperServiceProvider::createUserDefaultLists($user->id);
            
            // Final verification
            $finalUser = User::where('email', $email)->first();
            if ($finalUser) {
                \Log::info('Final admin verification - User ID: ' . $finalUser->id . ', Role ID: ' . $finalUser->role_id);
                
                if ($finalUser->role_id == 1) {
                    \Log::info('✅ Admin user setup completed successfully!');
                } else {
                    \Log::warning('Admin user created but role may not be set correctly. Role ID: ' . $finalUser->role_id);
                }
            } else {
                \Log::error('Could not find admin user after creation');
            }
            
        } catch (\Exception $e) {
            \Log::error('Failed to create admin user: ' . $e->getMessage());
            \Log::error('Error file: ' . $e->getFile() . ' line: ' . $e->getLine());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            
            // Don't throw - allow installation to complete but log the issue
            \Log::warning('Installation will continue but admin user creation may have failed');
        }

        // 4. Create an installed file over public dir
        Storage::disk('local')->put('installed', session()->get('license'));

        // 5. Create storage symlink
        Artisan::call('storage:link');

        // We could even do this to cache the config/avoid Xampp on Windows .env related issues
        // php artisan config:cache

        // FINAL VERIFICATION: Check that all critical data was saved to database
        \Log::info('=== RUNNING FINAL VERIFICATION ===');
        
        try {
            // Check site settings
            $siteNameSetting = DB::table('settings')->where('key', 'site.name')->value('value');
            $siteUrlSetting = DB::table('settings')->where('key', 'site.app_url')->value('value');
            $licenseSetting = DB::table('settings')->where('key', 'license.product_license_key')->value('value');
            
            \Log::info('Final verification - Site name in DB: ' . ($siteNameSetting ?: 'NOT FOUND'));
            \Log::info('Final verification - Site URL in DB: ' . ($siteUrlSetting ?: 'NOT FOUND'));
            \Log::info('Final verification - License key in DB: ' . ($licenseSetting ?: 'NOT FOUND'));
            
            // Check admin user
            $adminUser = User::where('email', $email)->first();
            if ($adminUser) {
                \Log::info('Final verification - Admin user found: ' . $adminUser->email . ' (ID: ' . $adminUser->id . ', Role: ' . $adminUser->role_id . ')');
            } else {
                \Log::error('Final verification - Admin user NOT FOUND in database!');
            }
            
            // Check if installed file was created
            $installedFileExists = Storage::disk('local')->exists('installed');
            \Log::info('Final verification - Installed file exists: ' . ($installedFileExists ? 'YES' : 'NO'));
            
            if (!$siteNameSetting || !$siteUrlSetting || !$licenseSetting || !$adminUser || !$installedFileExists) {
                \Log::warning('Some installation components may not have been saved correctly');
            } else {
                \Log::info('=== INSTALLATION COMPLETED SUCCESSFULLY ===');
            }
            
        } catch (\Exception $e) {
            \Log::error('Final verification failed: ' . $e->getMessage());
        }

        return view('installer.finish');
    }

    /**
     * Renders upgrade view.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function upgrade()
    {
        if(!(Auth::check() && Auth::user()->role_id == 1)){
            return redirect('/'); // Direct URL instead of route() to avoid APP_URL dependency
        }
        app()->instance('pretend_migration', true); // Register a flag
        $canMigrate = InstallerServiceProvider::hasAvailableMigrations();

        return view('installer.upgrade', ['canMigrate'=>$canMigrate]);
    }

    /**
     * Runs the update.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function doUpgrade()
    {
        try {
            Artisan::call('down');
            Artisan::call('migrate', ['--force'=>true]);
            Artisan::call('up');
        } catch (\Exception $e) {
            Redirect::to('/update')->with('error', $e->getMessage());
        }

        return Redirect::to('/update')->with('success', __('Database updated successfully.'));
    }

    /**
     * Saves APP_URL to .env file.
     * @param string $app_url
     */
    private function saveAppUrlToEnv($app_url) {
        try {
            // Parse the URL to get the host
            $parsedUrl = parse_url($app_url);
            $host = $parsedUrl['host'] ?? '';
            
            // Block ALL localhost URLs (with or without ports) to prevent redirect issues
            // Only allow legitimate domain URLs to be saved to .env
            if (strpos($host, 'localhost') !== false || strpos($host, '127.0.0.1') !== false) {
                \Log::info('Skipping APP_URL save to .env for localhost URL to prevent redirect issues: ' . $app_url);
                return true; // Return success but don't save localhost URLs
            }
            
            \Log::info('Saving APP_URL to .env: ' . $app_url . ' (host: ' . $host . ')');
            
            $envFile = base_path('.env');
            
            if (!file_exists($envFile)) {
                \Log::warning('.env file does not exist at: ' . $envFile);
                return false;
            }
            
            // Read current .env content
            $content = file_get_contents($envFile);
            \Log::info('Current .env file content length: ' . strlen($content));
            
            // Remove any existing APP_URL lines (handles various formats)
            $content = preg_replace('/^APP_URL\s*=.*$/mi', '', $content);
            
            // Remove trailing blank lines
            $content = rtrim($content, "\r\n");
            
            // Add the new APP_URL
            $content .= "\n";
            $content .= 'APP_URL="' . $app_url . '"';
            $content .= "\n";
            
            // Write back to .env file
            $result = file_put_contents($envFile, $content);
            
            if ($result === false) {
                \Log::error('Failed to write to .env file: ' . $envFile);
                return false;
            }
            
            // Verify the change
            $verifyContent = file_get_contents($envFile);
            if (strpos($verifyContent, 'APP_URL="' . $app_url . '"') !== false) {
                \Log::info('Successfully saved APP_URL to .env file: ' . $app_url);
                
                // Also try the old append method as backup
                try {
                    // Clean up any potential duplicates
                    $content = file_get_contents($envFile);
                    $lines = explode("\n", $content);
                    $cleanLines = [];
                    $appUrlFound = false;
                    
                    foreach ($lines as $line) {
                        if (strpos($line, 'APP_URL') === 0) {
                            if (!$appUrlFound) {
                                $cleanLines[] = 'APP_URL="' . $app_url . '"';
                                $appUrlFound = true;
                            }
                        } else {
                            $cleanLines[] = $line;
                        }
                    }
                    
                    if (count($cleanLines) > 0) {
                        file_put_contents($envFile, implode("\n", $cleanLines));
                    }
                    
                } catch (\Exception $e) {
                    \Log::warning('Backup method failed but main save succeeded: ' . $e->getMessage());
                }
                
                return true;
            } else {
                \Log::error('APP_URL was not saved correctly to .env file. Current content: ' . $verifyContent);
                return false;
            }
            
        } catch (\Exception $e) {
            \Log::error('Exception while saving APP_URL to .env: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Saves db info to env file with proper handling.
     * @param $request
     */
    public function saveEnvValues($request) {
        if(!InstallerServiceProvider::getLockCode()){
            \Log::error('Lock code check failed');
            return false;
        }

        try {
            $envFile = base_path('.env');
            
            if (!file_exists($envFile)) {
                \Log::error('.env file does not exist: ' . $envFile);
                return false;
            }

            // Read current .env content
            $content = file_get_contents($envFile);
            \Log::info('Current .env content length: ' . strlen($content));
            
            // Remove existing database-related lines to prevent duplicates
            $lines = explode("\n", $content);
            $cleanLines = [];
            
            foreach ($lines as $line) {
                $line = trim($line);
                // Skip empty lines and lines that start with DB_
                if (!empty($line) && strpos($line, 'DB_') !== 0) {
                    $cleanLines[] = $line;
                }
            }
            
            // Add database credentials
            $cleanLines[] = 'DB_HOST="' . $request->session()->get('db_host') . '"';
            $cleanLines[] = 'DB_DATABASE="' . $request->session()->get('db_name') . '"';
            $cleanLines[] = 'DB_USERNAME="' . $request->session()->get('db_username') . '"';
            $cleanLines[] = 'DB_PORT="' . $request->session()->get('db_port') . '"';
            $cleanLines[] = 'DB_PASSWORD="' . $request->session()->get('db_password') . '"';
            
            // Write cleaned content back to .env
            $newContent = implode("\n", $cleanLines) . "\n";
            $result = file_put_contents($envFile, $newContent);
            
            if ($result === false) {
                \Log::error('Failed to write to .env file: ' . $envFile);
                return false;
            }
            
            \Log::info('Database credentials saved to .env successfully');
            
            // CRITICAL: Clear config cache to force Laravel to read new .env values
            \Log::info('Clearing config cache...');
            Artisan::call('config:clear');
            
            // Verify .env file contains the database credentials
            $verifyContent = file_get_contents($envFile);
            $dbHostInEnv = strpos($verifyContent, 'DB_HOST=') !== false;
            $dbNameInEnv = strpos($verifyContent, 'DB_DATABASE=') !== false;
            
            if (!$dbHostInEnv || !$dbNameInEnv) {
                \Log::error('Database credentials verification failed');
                \Log::error('Expected DB_HOST, got: ' . substr($verifyContent, 0, 500));
                return false;
            }
            
            \Log::info('Database credentials verified in .env file');
            return true;
            
        } catch (\Exception $e) {
            \Log::error('Exception while saving database credentials to .env: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            return false;
        }
    }
}
